/*
 * Decompiled with CFR 0.152.
 */
package edu.colorado.phet.glaciers.model;

import edu.colorado.phet.common.phetcommon.math.PolarCartesianConverter;
import edu.colorado.phet.common.phetcommon.math.Vector2D;
import edu.colorado.phet.common.phetcommon.model.clock.ClockAdapter;
import edu.colorado.phet.common.phetcommon.model.clock.ClockEvent;
import edu.colorado.phet.glaciers.model.Climate;
import edu.colorado.phet.glaciers.model.Valley;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Iterator;

public class Glacier
extends ClockAdapter {
    private final Valley _valley;
    private final Climate _climate;
    private final Climate.ClimateListener _climateListener;
    private final ArrayList _listeners;
    private double _qela;
    private double _qelax;
    private double _glacierLength;
    private double _maxThickness;
    private double _averageIceThicknessSquares;
    private boolean _steadyState;
    private final Point2D _terminus;
    private Point2D _surfaceAtELA;
    private double _accelerationM = 0.03149378058107645;
    private double _accelerationB = -118.67636620809051;
    private final EvolutionState _evolutionState = new EvolutionState();

    public Glacier(Valley valley, Climate climate) {
        this._valley = valley;
        assert (this._valley.getHeadwallPositionReference().getX() == 0.0);
        this._climate = climate;
        this._climateListener = new Climate.ClimateListener(){

            public void temperatureChanged() {
                Glacier.this.handleClimateChange();
            }

            public void snowfallChanged() {
                Glacier.this.handleClimateChange();
            }
        };
        this._climate.addClimateListener(this._climateListener);
        this._listeners = new ArrayList();
        this._terminus = new Point2D.Double();
        this._surfaceAtELA = null;
        this.setSteadyState();
    }

    private void handleClimateChange() {
        this.updateIceThickness();
        if (this._steadyState) {
            this._steadyState = false;
            this.notifySteadyStateChanged();
        }
    }

    public Valley getValley() {
        return this._valley;
    }

    public Climate getClimate() {
        return this._climate;
    }

    public boolean isSteadyState() {
        return this._steadyState;
    }

    public void setSteadyState() {
        if (!this._steadyState) {
            double d = this._valley.getMaxElevation();
            this._qela = Math.min(this._climate.getELA(), d);
            this.updateIceThickness();
            this._steadyState = true;
            this.notifySteadyStateChanged();
            assert (this._qela <= d);
        }
    }

    public double getLength() {
        return this._glacierLength;
    }

    public double getHeadwallX() {
        return this._valley.getHeadwallPositionReference().getX();
    }

    public double getHeadwallY() {
        return this._valley.getHeadwallPositionReference().getY();
    }

    public double getTerminusX() {
        return this._terminus.getX();
    }

    public double getTerminusY() {
        return this._terminus.getY();
    }

    public Point2D getSurfaceAtELAReference() {
        return this._surfaceAtELA;
    }

    public double getSurfaceElevation(double d) {
        return this._valley.getElevation(d) + this.getIceThickness(d);
    }

    public void debug_setAccelerationM(double d) {
        System.out.println("Glacier.debug_setAccelerationM " + d);
        this._accelerationM = d;
    }

    public double debug_getAccelerationM() {
        return this._accelerationM;
    }

    public void debug_setAccelerationB(double d) {
        System.out.println("Glacier.debug_setAccelerationB " + d);
        this._accelerationB = d;
    }

    public double debug_getAccelerationB() {
        return this._accelerationB;
    }

    public double getIceThickness(double d) {
        double d2 = 0.0;
        double d3 = this._valley.getHeadwallPositionReference().getX();
        if (d > d3 && d < this._terminus.getX()) {
            double d4 = d3 + 0.5 * this._glacierLength;
            double d5 = Math.max(42.0 - 0.01 * this._qela, 1.5);
            double d6 = 1.5 * d4;
            double d7 = Math.pow(d4, d5);
            if (d < d4) {
                d2 = Math.sqrt(d6 * d6 - (d - d4) * (d - d4)) * (this._maxThickness / d6);
                d2 *= (d7 - Math.pow(Math.abs(d - d4), d5)) / d7;
            } else if (d < this._terminus.getX()) {
                d2 = Math.sqrt(d4 * d4 - (d - d4) * (d - d4)) * (this._maxThickness / d4);
            }
        }
        assert (d2 >= 0.0);
        return d2;
    }

    private void updateIceThickness() {
        this._surfaceAtELA = null;
        this._averageIceThicknessSquares = 0.0;
        this._qelax = Glacier.computeQuasiELAX(this._qela);
        this._glacierLength = this._qelax / 0.6;
        assert (this._glacierLength >= 0.0);
        this._maxThickness = 2.3 * Math.sqrt(this._qelax);
        assert (this._maxThickness >= 0.0);
        if (this._glacierLength == 0.0) {
            this._terminus.setLocation(this._valley.getHeadwallPositionReference());
        } else {
            double d = this._climate.getELA();
            double d2 = this._valley.getHeadwallPositionReference().getX();
            double d3 = this._valley.getHeadwallPositionReference().getY();
            double d4 = this._valley.getMaxElevation();
            double d5 = d2 + this._glacierLength;
            double d6 = this._valley.getElevation(d5);
            this._terminus.setLocation(d5, d6);
            assert (this._terminus.getX() >= d2);
            assert (this._terminus.getY() <= d3);
            double d7 = d2 + 80.0;
            double d8 = 0.0;
            double d9 = 0.0;
            double d10 = 0.0;
            double d11 = 0.0;
            boolean bl = false;
            while (!bl) {
                d8 = this.getIceThickness(d7);
                if (d8 == 0.0) {
                    bl = true;
                    continue;
                }
                d9 += d8 * d8;
                d10 += 1.0;
                if (this._surfaceAtELA == null && d < d4 && (d11 = this._valley.getElevation(d7) + d8) <= d) {
                    this._surfaceAtELA = this.findSurfaceAtELA(d, d7 - 80.0, d7);
                }
                d7 += 80.0;
            }
            if (this._surfaceAtELA == null && d < d4 && d6 < d) {
                this._surfaceAtELA = this.findSurfaceAtELA(d, d5 - 80.0, d5);
            }
            if (d10 > 0.0) {
                this._averageIceThicknessSquares = d9 / d10;
            }
            assert (this._averageIceThicknessSquares >= 0.0);
        }
        this.notifyIceThicknessChanged();
    }

    private static double computeQuasiELAX(double d) {
        double d2;
        if (d < 3800.0) {
            d2 = d * -37.57 + 157076.0;
        } else if (d < 4150.0) {
            double d3 = d - 3800.0;
            d2 = 14300.0 + -37.57 * d3 + -0.098411 * d3 * d3 + 2.81185E-4 * d3 * d3 * d3;
        } else {
            d2 = d * -3.12489 + 14118.0;
        }
        if (d2 < 0.0) {
            d2 = 0.0;
        }
        assert (d2 >= 0.0);
        return d2;
    }

    private Point2D findSurfaceAtELA(double d, double d2, double d3) {
        assert (d2 < d3);
        Point2D.Double double_ = null;
        double d4 = -1.0;
        double d5 = -1.0;
        for (double d6 = d2; d6 <= d3 && double_ == null; d6 += 1.0) {
            double d7 = this.getSurfaceElevation(d6);
            double d8 = d7 - d;
            if (Math.abs(d8) <= 1.0) {
                double_ = new Point2D.Double(d6, d7);
            } else if (d8 < 0.0) {
                double_ = d4 == -1.0 ? new Point2D.Double(d6, d7) : (d8 < d5 ? new Point2D.Double(d6, d7) : new Point2D.Double(d6, d4));
            }
            d4 = d7;
            d5 = d8;
        }
        assert (double_ != null);
        return double_;
    }

    public Vector2D getIceVelocity(double d, double d2, Vector2D vector2D) {
        double d3 = this.getIceSpeed(d, d2);
        double d4 = this._valley.getDirection(d, d + 80.0);
        double d5 = PolarCartesianConverter.getX(d3, d4);
        double d6 = PolarCartesianConverter.getY(d3, d4);
        vector2D.setComponents(d5, d6);
        return vector2D;
    }

    public Vector2D getIceVelocity(double d, double d2) {
        return this.getIceVelocity(d, d2, new Vector2D());
    }

    private double getIceSpeed(double d, double d2) {
        double d3 = 0.0;
        double d4 = this.getIceThickness(d);
        if (d4 > 0.0) {
            double d5 = this._valley.getElevation(d);
            double d6 = d5 + d4;
            if (d2 >= d5 && d2 <= d6) {
                double d7 = (d2 - d5) / d4;
                double d8 = Glacier.computeVerticallyAveragedDeformationIceSpeed(d4, this._averageIceThicknessSquares);
                d3 = 20.0 + d8 * 5.0 * (d7 - 1.5 * d7 * d7 + d7 * d7 * d7 - 0.25 * d7 * d7 * d7 * d7);
            }
        }
        assert (d3 >= 0.0);
        return d3;
    }

    private static double computeVerticallyAveragedDeformationIceSpeed(double d, double d2) {
        double d3 = 0.0;
        if (d2 != 0.0) {
            d3 = d * d * 20.0 / d2;
        }
        assert (d3 >= 0.0);
        return d3;
    }

    public EvolutionState debug_getEvolutionState() {
        return this._evolutionState;
    }

    public void clockTicked(ClockEvent clockEvent) {
        if (!this.isSteadyState()) {
            double d = clockEvent.getSimulationTimeChange();
            double d2 = this._climate.getELA();
            double d3 = this.getTimescale();
            double d4 = this._valley.getMaxElevation();
            double d5 = this._glacierLength;
            double d6 = (d2 - this._qela) * (1.0 - Math.exp(-d / d3));
            d6 = d2 < this._qela ? Math.max(d6, d * -2.0) : Math.min(d6, d * 4.0);
            if (this._qela > 4150.0) {
                d6 = d6 * -37.57 / -3.12489;
            } else if (this._qela > 3800.0) {
                d6 *= this._qela * this._accelerationM + this._accelerationB;
            }
            this._qela += d6;
            if (this._qela > d4) {
                this._qela = d4;
                if (d2 > d4) {
                    this.setSteadyState();
                }
            } else if (Math.abs(d2 - this._qela) <= 0.1) {
                this.setSteadyState();
            } else {
                this.updateIceThickness();
            }
            this._evolutionState.ela = d2;
            this._evolutionState.timescale = d3;
            this._evolutionState.deltaQela = d6;
            this._evolutionState.qela = this._qela;
            this._evolutionState.qelax = this._qelax;
            this._evolutionState.glacierLength = this._glacierLength;
            this._evolutionState.deltaLength = this._glacierLength - d5;
            this._evolutionState.terminus.setLocation(this._terminus);
            assert (this._qela <= d4);
        }
    }

    private double getTimescale() {
        double d = -0.22 * this._climate.getELA() + 1026.0;
        if (d < 50.0) {
            d = 50.0;
        } else if (d > 200.0) {
            d = 200.0;
        }
        return d;
    }

    public void addGlacierListener(GlacierListener glacierListener) {
        this._listeners.add(glacierListener);
    }

    public void removeGlacierListener(GlacierListener glacierListener) {
        this._listeners.remove(glacierListener);
    }

    private void notifyIceThicknessChanged() {
        ArrayList arrayList = new ArrayList(this._listeners);
        Iterator iterator = arrayList.iterator();
        while (iterator.hasNext()) {
            ((GlacierListener)iterator.next()).iceThicknessChanged();
        }
    }

    private void notifySteadyStateChanged() {
        Iterator iterator = this._listeners.iterator();
        while (iterator.hasNext()) {
            ((GlacierListener)iterator.next()).steadyStateChanged();
        }
    }

    public static class EvolutionState {
        public double ela;
        public double timescale;
        public double deltaQela;
        public double qela;
        public double qelax;
        public double glacierLength;
        public double deltaLength;
        public Point2D terminus = new Point2D.Double();

        public void setState(EvolutionState evolutionState) {
            this.ela = evolutionState.ela;
            this.timescale = evolutionState.timescale;
            this.deltaQela = evolutionState.deltaQela;
            this.qela = evolutionState.qela;
            this.qelax = evolutionState.qelax;
            this.glacierLength = evolutionState.glacierLength;
            this.deltaLength = evolutionState.deltaLength;
            this.terminus.setLocation(evolutionState.terminus);
        }
    }

    public static class GlacierAdapter
    implements GlacierListener {
        public void iceThicknessChanged() {
        }

        public void steadyStateChanged() {
        }
    }

    public static interface GlacierListener {
        public void iceThicknessChanged();

        public void steadyStateChanged();
    }
}

